import { tradeHistory } from '@/lib/data';
import type { Trade } from '@/lib/types';
import { EventEmitter } from 'events';

type AccountState = {
    balance: number;
    equity: number;
    profit: number;
    dailyDrawdown: number;
    maxDrawdown: number;
    openTrades: Trade[];
};

type Market = {
    [symbol: string]: {
        price: number;
        spread: number;
    };
}

export class TradeSimulator extends EventEmitter {
    private accountState: AccountState;
    private tradeHistory: Trade[] = [];
    private simulationInterval: NodeJS.Timeout | null = null;
    private market: Market = {
        'EURUSD': { price: 1.075, spread: 0.0001 },
        'GBPUSD': { price: 1.25, spread: 0.00015 },
        'USDJPY': { price: 155, spread: 0.015 },
        'AUDUSD': { price: 0.66, spread: 0.00012 },
        'USDCAD': { price: 1.37, spread: 0.00018 },
    };
    private lastNotification: { title: string, description: string } | null = null;
    private startCapital: number;
    private static instance: TradeSimulator;
    private peakEquity: number;
    private dailyPeakEquity: number;
    private lastDay: number;


    private constructor(startCapital: number) {
        super();
        this.startCapital = startCapital;
        this.peakEquity = startCapital;
        this.dailyPeakEquity = startCapital;
        this.lastDay = new Date().getDate();

        this.accountState = {
            balance: startCapital,
            equity: startCapital,
            profit: 0,
            dailyDrawdown: 0,
            maxDrawdown: 0,
            openTrades: [],
        };
        this.initializeTrades();
    }

    public static getInstance(startCapital: number): TradeSimulator {
        if (!TradeSimulator.instance) {
            TradeSimulator.instance = new TradeSimulator(startCapital);
        }
        return TradeSimulator.instance;
    }


    private initializeTrades() {
        // Open a few trades to start
        this.openTrade('EURUSD', 'Buy', 1.0);
        setTimeout(() => this.openTrade('GBPUSD', 'Sell', 0.5), 1000);
    }

    public start() {
        if (this.simulationInterval) return;
        this.simulationInterval = setInterval(() => this.tick(), 1500); // Faster ticks
    }

    public stop() {
        if (this.simulationInterval) {
            clearInterval(this.simulationInterval);
            this.simulationInterval = null;
        }
    }

    private tick() {
        const currentDate = new Date();
        if (currentDate.getDate() !== this.lastDay) {
            this.dailyPeakEquity = this.accountState.equity;
            this.lastDay = currentDate.getDate();
        }

        // Simulate market price changes with more volatility
        for (const symbol in this.market) {
            const volatility = 0.0005; // Base volatility
            const priceChange = (Math.random() - 0.49) * this.market[symbol].price * volatility;
            this.market[symbol].price += priceChange;
        }

        this.updateAccountState();

        // Simulate closing a trade occasionally
        if (this.accountState.openTrades.length > 0 && Math.random() < 0.08) {
            const tradeToClose = this.accountState.openTrades[Math.floor(Math.random() * this.accountState.openTrades.length)];
            this.closeTrade(tradeToClose.id);
        }

        // Simulate opening a new trade occasionally
        if (this.accountState.openTrades.length < 5 && Math.random() < 0.04) {
            const symbols = Object.keys(this.market);
            const symbol = symbols[Math.floor(Math.random() * symbols.length)];
            const type = Math.random() < 0.5 ? 'Buy' : 'Sell';
            const volume = Math.round((Math.random() * 1.5 + 0.1) * 100) / 100;
            this.openTrade(symbol, type, volume);
        }


        this.emit('tick');
    }
    
    private updateAccountState() {
        let openTradesProfit = 0;
        this.accountState.openTrades.forEach(trade => {
            const currentPrice = this.market[trade.symbol].price;
            let tradeProfit = 0;
            const pointValue = trade.symbol.includes('JPY') ? 1000 : 100000;
            if (trade.type === 'Buy') {
                tradeProfit = (currentPrice - trade.openPrice) * trade.volume * pointValue;
            } else { // Sell
                tradeProfit = (trade.openPrice - currentPrice) * trade.volume * pointValue;
            }
            trade.profit = tradeProfit;
            openTradesProfit += tradeProfit;
        });

        const closedTradesProfit = this.tradeHistory.reduce((sum, trade) => sum + trade.profit, 0);

        this.accountState.equity = this.accountState.balance + openTradesProfit;
        this.accountState.profit = closedTradesProfit + openTradesProfit;

        // Update drawdowns
        this.peakEquity = Math.max(this.peakEquity, this.accountState.equity);
        this.dailyPeakEquity = Math.max(this.dailyPeakEquity, this.accountState.equity);

        this.accountState.maxDrawdown = this.peakEquity - this.accountState.equity;
        this.accountState.dailyDrawdown = this.dailyPeakEquity - this.accountState.equity;
    }


    public openTrade(symbol: string, type: 'Buy' | 'Sell', volume: number) {
        const marketData = this.market[symbol];
        if (!marketData) return;

        const openPrice = type === 'Buy' ? marketData.price + marketData.spread / 2 : marketData.price - marketData.spread / 2;

        const newTrade: Trade = {
            id: `T${Date.now()}-${Math.random().toString(36).substring(2, 8)}`,
            symbol,
            type,
            volume,
            openTime: new Date().toISOString(),
            openPrice,
            closeTime: '',
            closePrice: 0,
            profit: 0,
        };

        this.accountState.openTrades.push(newTrade);
        this.emit('tradeOpened', newTrade);
        this.updateAccountState();
        this.emit('tick');
    }

    public closeTrade(tradeId: string) {
        const tradeIndex = this.accountState.openTrades.findIndex(t => t.id === tradeId);
        if (tradeIndex === -1) return;

        const trade = this.accountState.openTrades[tradeIndex];
        const marketData = this.market[trade.symbol];
        const closePrice = trade.type === 'Buy' ? marketData.price - marketData.spread / 2 : marketData.price + marketData.spread / 2;
        
        // Final profit calculation is done here
        const pointValue = trade.symbol.includes('JPY') ? 1000 : 100000;
        let profit = 0;
        if (trade.type === 'Buy') {
            profit = (closePrice - trade.openPrice) * trade.volume * pointValue;
        } else {
            profit = (trade.openPrice - closePrice) * trade.volume * pointValue;
        }
        
        const closedTrade: Trade = { ...trade, closePrice, closeTime: new Date().toISOString(), profit };

        this.accountState.balance += profit;
        this.accountState.openTrades.splice(tradeIndex, 1);
        this.tradeHistory.push(closedTrade);
        
        this.emit('tradeClosed', closedTrade);
        this.updateAccountState();
        this.emit('tick');
    }

    public getAccountState(): AccountState {
        this.updateAccountState();
        return {
            ...this.accountState,
        };
    }
    
    public getMarketData(): Market {
        return this.market;
    }

    public getTradeHistory(): Trade[] {
        // Return a combined list of closed and open trades for history display purposes
        return [...this.tradeHistory].sort((a, b) => new Date(b.openTime).getTime() - new Date(a.openTime).getTime());
    }

    public getOpenTrades(): Trade[] {
        return [...this.accountState.openTrades];
    }


    public getLastNotification() {
        return this.lastNotification;
    }

    public setLastNotification(notification: { title: string, description: string } | null) {
        this.lastNotification = notification;
    }
}
