
'use server';

import {
  enforceChallengeRules,
  type EnforceChallengeRulesInput,
  type EnforceChallengeRulesOutput,
} from '@/ai/flows/enforce-challenge-rules';
import {
    generateSwahiliChallengeNames,
    type GenerateSwahiliChallengeNamesInput,
    type GenerateSwahiliChallengeNamesOutput,
} from '@/ai/flows/generate-swahili-challenge-names';

const ZENOPAY_API_URL = 'https://zenoapi.com/api/payments/mobile_money_tanzania';
const ZENOPAY_STATUS_URL = 'https://zenoapi.com/api/payments/order-status';
const API_KEY = process.env.ZENOPAY_API_KEY || '';

export type ZenoPaymentInput = {
    buyer_email: string;
    buyer_name: string;
    buyer_phone: string;
    amount: number;
};

export async function initiateZenoPayment(input: ZenoPaymentInput) {
    const order_id = crypto.randomUUID();

    try {
        const response = await fetch(ZENOPAY_API_URL, {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
                'x-api-key': API_KEY,
            },
            body: JSON.stringify({
                order_id,
                ...input,
            }),
        });

        const data = await response.json();
        
        if (data.status === 'success') {
            return { success: true, order_id, message: data.message };
        } else {
            return { success: false, message: data.message || 'Payment initiation failed' };
        }
    } catch (error) {
        console.error('ZenoPay Initiation Error:', error);
        return { success: false, message: 'Internal server error during payment initiation' };
    }
}

export async function checkZenoPaymentStatus(order_id: string) {
    try {
        const response = await fetch(`${ZENOPAY_STATUS_URL}?order_id=${order_id}`, {
            method: 'GET',
            headers: {
                'x-api-key': API_KEY,
            },
        });

        const data = await response.json();

        if (data.result === 'SUCCESS' && data.data?.[0]?.payment_status === 'COMPLETED') {
            return { status: 'COMPLETED', data: data.data[0] };
        }

        return { status: data.data?.[0]?.payment_status || 'PENDING' };
    } catch (error) {
        console.error('ZenoPay Status Error:', error);
        return { status: 'ERROR' };
    }
}

export async function checkChallengeRules(input: EnforceChallengeRulesInput): Promise<EnforceChallengeRulesOutput> {
  try {
    const result = await enforceChallengeRules(input);
    return result;
  } catch (error) {
    console.error("Error in checkChallengeRules:", error);
    return {
      isViolatingRules: false,
      notificationMessage: "Error checking rules.",
    };
  }
}

export async function getSwahiliChallengeName(input: GenerateSwahiliChallengeNamesInput): Promise<GenerateSwahiliChallengeNamesOutput> {
    try {
        const result = await generateSwahiliChallengeNames(input);
        return result;
    } catch (error) {
        console.error("Error generating Swahili name:", error instanceof Error ? error.message : String(error));
        return {
            challengeName: input.challengeType,
        };
    }
}
