import {
  Card,
  CardContent,
  CardDescription,
  CardFooter,
  CardHeader,
  CardTitle,
} from '@/components/ui/card';
import { Button } from '@/components/ui/button';
import {
  Dialog,
  DialogContent,
  DialogDescription,
  DialogHeader,
  DialogTitle,
  DialogTrigger,
} from '@/components/ui/dialog';
import { getSwahiliChallengeName } from '@/lib/actions';
import type { Challenge } from '@/lib/types';
import { toast } from '@/hooks/use-toast';
import { CheckCircle } from 'lucide-react';
import Image from 'next/image';

interface ChallengeCardProps {
  challenge: Challenge;
}

const Stat = ({ label, value }: { label: string; value: string }) => (
  <div className="flex justify-between text-sm">
    <p className="text-muted-foreground">{label}</p>
    <p className="font-medium">{value}</p>
  </div>
);

function PaymentDialogContent() {
    const handlePayment = () => {
        toast({
            title: "Payment Successful!",
            description: "Your new challenge account has been created.",
            action: <CheckCircle className="text-green-500" />,
        });
    };

    return (
        <DialogContent>
            <DialogHeader>
            <DialogTitle>Complete Your Payment</DialogTitle>
            <DialogDescription>
                Choose your preferred payment method to start the challenge.
            </DialogDescription>
            </DialogHeader>
            <div className="space-y-4 py-4">
                <Button onClick={handlePayment} className="w-full justify-start gap-4 h-14 bg-green-500 hover:bg-green-600">
                    <Image data-ai-hint="logo" src="https://picsum.photos/40/40" alt="M-Pesa" width={40} height={40} className="rounded-md" />
                    Pay with M-Pesa
                </Button>
                <Button onClick={handlePayment} className="w-full justify-start gap-4 h-14 bg-purple-500 hover:bg-purple-600">
                    <Image data-ai-hint="logo" src="https://picsum.photos/40/40" alt="Tigo Pesa" width={40} height={40} className="rounded-md" />
                    Pay with Tigo Pesa
                </Button>
                <Button onClick={handlePayment} className="w-full justify-start gap-4 h-14 bg-red-500 hover:bg-red-600">
                    <Image data-ai-hint="logo" src="https://picsum.photos/40/40" alt="Airtel Money" width={40} height={40} className="rounded-md" />
                    Pay with Airtel Money
                </Button>
            </div>
        </DialogContent>
    )
}

export async function ChallengeCard({ challenge }: ChallengeCardProps) {
  const { challengeName } = await getSwahiliChallengeName({ challengeType: challenge.type });
  const formatCurrency = (value: number) =>
    new Intl.NumberFormat('en-TZ', { style: 'currency', currency: 'TZS', minimumFractionDigits: 0 }).format(value);

  return (
    <Card className="flex flex-col">
      <CardHeader>
        <CardTitle className="font-headline text-primary">{challengeName}</CardTitle>
        <CardDescription>{challenge.type}</CardDescription>
      </CardHeader>
      <CardContent className="flex-grow space-y-4">
        <div className="text-3xl font-bold">{formatCurrency(challenge.startCapital)}</div>
        <div className="space-y-2">
          <Stat label="Profit Target" value={`${(challenge.profitTarget / challenge.startCapital) * 100}%`} />
          <Stat label="Max Drawdown" value={`${(challenge.maxDrawdownLimit / challenge.startCapital) * 100}%`} />
          <Stat label="Daily Drawdown" value={`${(challenge.dailyDrawdownLimit / challenge.startCapital) * 100}%`} />
        </div>
      </CardContent>
      <CardFooter>
        <Dialog>
            <DialogTrigger asChild>
                <Button className="w-full bg-accent hover:bg-accent/90 text-accent-foreground">
                    Start - {formatCurrency(challenge.fee)}
                </Button>
            </DialogTrigger>
            <PaymentDialogContent />
        </Dialog>
      </CardFooter>
    </Card>
  );
}
