
'use client';

import { DashboardOverview } from '@/components/dashboard-overview';
import { Card, CardContent, CardHeader, CardTitle, CardDescription } from '@/components/ui/card';
import { Button } from '@/components/ui/button';
import { PlusCircle, ArrowUpRight, History, BarChart3, Users, Zap, Globe } from 'lucide-react';
import Link from 'next/link';
import { useUser, useCollection, useFirestore } from '@/firebase';
import { query, collection, where, orderBy, limit } from 'firebase/firestore';
import { Skeleton } from '@/components/ui/skeleton';

export default function DashboardPage() {
  const { user } = useUser();
  const db = useFirestore();

  const transactionsQuery = user && db 
    ? query(collection(db, 'transactions'), where('userId', '==', user.uid), orderBy('timestamp', 'desc'), limit(5))
    : null;

  const { data: transactions, loading } = useCollection(transactionsQuery);

  const formatCurrency = (value: number) =>
    new Intl.NumberFormat('en-TZ', { style: 'currency', currency: 'TZS', minimumFractionDigits: 0 }).format(value);

  const totalRevenue = transactions?.reduce((acc, curr: any) => acc + (curr.amount > 0 ? curr.amount : 0), 0) || 0;

  return (
    <div className="space-y-8">
      <div className="flex flex-col md:flex-row md:items-center justify-between gap-4">
        <div>
          <h1 className="text-3xl font-bold font-headline">Karibu, {user?.displayName?.split(' ')[0] || 'Creator'}</h1>
          <p className="text-muted-foreground">Your SaaS Monetization Hub is live.</p>
        </div>
        <div className="flex gap-3">
          <Link href="/create">
            <Button className="rounded-2xl h-12 px-6 gap-2 bg-primary hover:bg-primary/90">
              <PlusCircle className="h-5 w-5" />
              Monetize Content
            </Button>
          </Link>
          <Button variant="outline" className="rounded-2xl h-12 px-6 gap-2">
            <Zap className="h-5 w-5 text-primary" />
            API Keys
          </Button>
        </div>
      </div>

      <DashboardOverview 
        balance={totalRevenue * 0.9} // Simulated split
        totalSales={transactions?.length || 0}
        totalRevenue={totalRevenue}
        activeFollowers={Math.floor(totalRevenue / 1000) || 0}
      />

      <div className="grid grid-cols-1 lg:grid-cols-3 gap-6">
        <Card className="lg:col-span-2 rounded-[2.5rem] border-none shadow-xl shadow-primary/5 bg-card/60">
          <CardHeader className="flex flex-row items-center justify-between">
            <div>
              <CardTitle className="font-headline text-2xl">Sales Velocity</CardTitle>
              <CardDescription>Your infrastructure conversion real-time feed.</CardDescription>
            </div>
            <Link href="/wallet">
               <Button variant="ghost" size="sm" className="rounded-full">Full Report</Button>
            </Link>
          </CardHeader>
          <CardContent>
            {loading ? (
                <div className="space-y-4">
                    {[1, 2, 3].map(i => <Skeleton key={i} className="h-16 w-full rounded-2xl" />)}
                </div>
            ) : (
                <div className="space-y-6">
                {transactions?.map((tx: any) => (
                    <div key={tx.id} className="flex items-center justify-between p-2 hover:bg-primary/5 rounded-2xl transition-colors">
                    <div className="flex items-center gap-4">
                        <div className="h-12 w-12 rounded-2xl bg-primary/10 flex items-center justify-center text-primary">
                        <Zap className="h-6 w-6 fill-primary/20" />
                        </div>
                        <div>
                        <p className="font-bold text-base">{tx.description || 'Micro-Unlock'}</p>
                        <p className="text-xs text-muted-foreground">{tx.timestamp?.toDate ? tx.timestamp.toDate().toLocaleString() : 'Just now'}</p>
                        </div>
                    </div>
                    <div className="text-right">
                        <div className="font-bold text-lg text-primary">+{formatCurrency(tx.amount)}</div>
                        <div className="text-[10px] font-bold uppercase tracking-wider text-muted-foreground">M-PESA COMPLETED</div>
                    </div>
                    </div>
                ))}
                {(!transactions || transactions.length === 0) && (
                    <div className="py-20 text-center space-y-4">
                        <BarChart3 className="h-12 w-12 text-muted-foreground/20 mx-auto" />
                        <p className="text-muted-foreground">Waiting for your first unlock...</p>
                    </div>
                )}
                </div>
            )}
          </CardContent>
        </Card>

        <div className="space-y-6">
            <Card className="rounded-[2.5rem] border-none shadow-xl shadow-primary/5 bg-primary text-primary-foreground overflow-hidden relative">
                <div className="absolute top-0 right-0 w-32 h-32 bg-white/10 rounded-full blur-3xl -mr-16 -mt-16"></div>
                <CardHeader>
                    <CardTitle className="font-headline">Quick Actions</CardTitle>
                    <CardDescription className="text-primary-foreground/80 font-medium">Infrastructure shortcuts.</CardDescription>
                </CardHeader>
                <CardContent className="space-y-3 relative z-10">
                    <Button variant="secondary" className="w-full justify-start h-14 rounded-2xl gap-3 text-base shadow-sm">
                        <PlusCircle className="h-6 w-6" />
                        Monetize Course
                    </Button>
                    <Button variant="secondary" className="w-full justify-start h-14 rounded-2xl gap-3 text-base shadow-sm">
                        <PlusCircle className="h-6 w-6" />
                        Lock Software/Link
                    </Button>
                    <Button variant="secondary" className="w-full justify-start h-14 rounded-2xl gap-3 text-base shadow-sm">
                        <Globe className="h-6 w-6" />
                        Preview Storefront
                    </Button>
                </CardContent>
            </Card>
            
            <Card className="rounded-[2.5rem] border-none shadow-xl shadow-primary/5 bg-card/60 p-8 space-y-4 border border-primary/10">
                <div className="flex items-center gap-3 font-bold text-primary">
                    <Users className="h-5 w-5" />
                    Fan Retention
                </div>
                <div className="space-y-2">
                    <div className="flex justify-between text-sm font-medium">
                        <span>Unlock Repeat Rate</span>
                        <span>84%</span>
                    </div>
                    <div className="h-2 w-full bg-primary/10 rounded-full">
                        <div className="h-full w-[84%] bg-primary rounded-full"></div>
                    </div>
                </div>
            </Card>
        </div>
      </div>
    </div>
  );
}
