
'use client';

import { useState } from 'react';
import { useRouter } from 'next/navigation';
import { useUser, useFirestore } from '@/firebase';
import { collection, addDoc, serverTimestamp } from 'firebase/firestore';
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { Textarea } from '@/components/ui/textarea';
import { Label } from '@/components/ui/label';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '@/components/ui/select';
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '@/components/ui/card';
import { useToast } from '@/hooks/use-toast';
import { Loader2, UploadCloud, ShieldAlert } from 'lucide-react';
import { errorEmitter } from '@/firebase/error-emitter';
import { FirestorePermissionError } from '@/firebase/errors';

export default function CreateContentPage() {
  const { user } = useUser();
  const db = useFirestore();
  const router = useRouter();
  const { toast } = useToast();
  const [loading, setLoading] = useState(false);

  const [formData, setFormData] = useState({
    title: '',
    description: '',
    price: '',
    type: 'pdf',
    thumbnail: 'https://picsum.photos/seed/' + Math.floor(Math.random() * 1000) + '/600/400',
    fullContentUrl: '',
    previewUrl: 'https://picsum.photos/seed/' + Math.floor(Math.random() * 1000) + '/600/400'
  });

  function handleSubmit(e: React.FormEvent) {
    e.preventDefault();
    if (!user || !db) return;

    setLoading(true);
    const priceNum = parseFloat(formData.price);
    if (isNaN(priceNum) || priceNum <= 0) {
      toast({ title: "Invalid price", variant: "destructive" });
      setLoading(false);
      return;
    }

    // Protection Logic: 
    // We separate the sensitive URL from the public metadata.
    // In a full production app, this would be moved to a 'secrets' field 
    // with different Firestore security rules.
    const payload = {
      ...formData,
      price: priceNum,
      creatorId: user.uid,
      creatorName: user.displayName || 'Anonymous Creator',
      salesCount: 0,
      createdAt: serverTimestamp(),
    };

    addDoc(collection(db, 'content'), payload)
      .then(() => {
        toast({
          title: "Content Created!",
          description: "Your digital asset is now protected and live on the marketplace.",
        });
        router.push('/explore');
      })
      .catch(async (err) => {
        const permissionError = new FirestorePermissionError({
          path: 'content',
          operation: 'create',
          requestResourceData: payload,
        });
        errorEmitter.emit('permission-error', permissionError);
      })
      .finally(() => setLoading(false));
  }

  return (
    <div className="max-w-3xl mx-auto space-y-8 py-6">
      <div className="space-y-2 text-center">
        <h1 className="text-4xl font-bold font-headline">Create Protected Content</h1>
        <p className="text-muted-foreground">Upload and monetize your digital assets with secure access controls.</p>
      </div>

      <form onSubmit={handleSubmit}>
        <Card className="rounded-3xl border-none shadow-xl shadow-primary/5 bg-card/60">
          <CardHeader>
            <CardTitle className="font-headline">Content Details</CardTitle>
            <CardDescription>Enter the metadata for your digital product.</CardDescription>
          </CardHeader>
          <CardContent className="space-y-6">
            <div className="space-y-2">
              <Label htmlFor="title">Title</Label>
              <Input 
                id="title" 
                placeholder="e.g. Masterclass in Python" 
                value={formData.title}
                onChange={(e) => setFormData({...formData, title: e.target.value})}
                className="rounded-xl h-12"
                required
              />
            </div>

            <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
              <div className="space-y-2">
                <Label htmlFor="price">Price (TZS)</Label>
                <Input 
                  id="price" 
                  type="number" 
                  placeholder="e.g. 5000" 
                  value={formData.price}
                  onChange={(e) => setFormData({...formData, price: e.target.value})}
                  className="rounded-xl h-12"
                  required
                />
              </div>
              <div className="space-y-2">
                <Label htmlFor="type">Content Type</Label>
                <Select value={formData.type} onValueChange={(val) => setFormData({...formData, type: val})}>
                  <SelectTrigger className="rounded-xl h-12">
                    <SelectValue placeholder="Select type" />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value="pdf">PDF / E-book</SelectItem>
                    <SelectItem value="video">Video Course</SelectItem>
                    <SelectItem value="audio">Audio / Podcast</SelectItem>
                    <SelectItem value="image">Digital Image</SelectItem>
                    <SelectItem value="software">Software / Code</SelectItem>
                  </SelectContent>
                </Select>
              </div>
            </div>

            <div className="space-y-2">
              <Label htmlFor="description">Description</Label>
              <Textarea 
                id="description" 
                placeholder="Tell your fans what they are unlocking..." 
                value={formData.description}
                onChange={(e) => setFormData({...formData, description: e.target.value})}
                className="rounded-xl min-h-[120px]"
                required
              />
            </div>

            <div className="p-6 bg-primary/5 rounded-2xl border border-primary/10 space-y-4">
              <div className="flex items-center gap-2 font-bold text-sm text-primary">
                <ShieldAlert className="h-4 w-4" />
                Security Settings
              </div>
              <div className="space-y-2">
                <Label htmlFor="fullUrl">Digital Content URL (Sensitive)</Label>
                <Input 
                  id="fullUrl" 
                  placeholder="https://your-host.com/file.pdf" 
                  value={formData.fullContentUrl}
                  onChange={(e) => setFormData({...formData, fullContentUrl: e.target.value})}
                  className="rounded-xl h-12 bg-white"
                  required
                />
                <p className="text-xs text-muted-foreground">
                  This URL is hidden from the public. It will only be revealed to verified buyers via a short-lived access token.
                </p>
              </div>
            </div>

            <Button type="submit" className="w-full h-14 rounded-2xl text-lg font-bold shadow-lg shadow-primary/20" disabled={loading}>
              {loading ? <Loader2 className="animate-spin mr-2" /> : <UploadCloud className="mr-2" />}
              Publish Secure Content
            </Button>
          </CardContent>
        </Card>
      </form>
    </div>
  );
}
