'use client';

import { useState, useEffect, useCallback } from 'react';
import { TradeSimulator } from '@/services/trade-simulator';
import { challenges } from '@/lib/data';
import type { Trade } from '@/lib/types';
import { Card, CardContent, CardHeader, CardTitle, CardDescription } from '@/components/ui/card';
import { Table, TableBody, TableCell, TableHead, TableHeader, TableRow } from '@/components/ui/table';
import { Badge } from '@/components/ui/badge';
import { ScrollArea } from '@/components/ui/scroll-area';
import { Button } from '@/components/ui/button';

const challenge = challenges[0];

type Market = ReturnType<TradeSimulator['getMarketData']>;

export function TradeTerminal() {
  const [simulator] = useState(() => TradeSimulator.getInstance(challenge.startCapital));
  const [openTrades, setOpenTrades] = useState<Trade[]>([]);
  const [marketData, setMarketData] = useState<Market>({});
  const [isClient, setIsClient] = useState(false);

  useEffect(() => {
    setIsClient(true);
  }, []);

  const handleUpdate = useCallback(() => {
    setOpenTrades([...simulator.getOpenTrades()]); // Create new array to trigger re-render
    setMarketData({...simulator.getMarketData()});
  },[simulator]);

  useEffect(() => {
    simulator.on('tick', handleUpdate);
    simulator.start(); // Ensure simulator is running

    // Set initial data
    handleUpdate();

    return () => {
      simulator.off('tick', handleUpdate);
      // We don't stop the simulator here because it's a singleton
    };
  }, [simulator, handleUpdate]);

  const formatCurrency = (value: number) =>
    new Intl.NumberFormat('en-US', {
      style: 'currency',
      currency: 'USD', // Using USD for P/L display convention
      minimumFractionDigits: 2,
      maximumFractionDigits: 2
    }).format(value);
    
  const formatPrice = (value: number) => value.toFixed(5);
  
  if (!isClient) {
    return null; // or a skeleton loader
  }

  return (
    <div className="grid grid-cols-1 lg:grid-cols-3 gap-6">
      <div className="lg:col-span-2">
        <Card>
          <CardHeader>
            <CardTitle>Open Positions</CardTitle>
            <CardDescription>Live trades running in the simulation.</CardDescription>
          </CardHeader>
          <CardContent>
            <ScrollArea className="h-96">
                <Table>
                <TableHeader>
                    <TableRow>
                    <TableHead>Symbol</TableHead>
                    <TableHead>Type</TableHead>
                    <TableHead>Volume</TableHead>
                    <TableHead>Open Price</TableHead>
                    <TableHead>Current Price</TableHead>
                    <TableHead className="text-right">Unrealized P/L</TableHead>
                    <TableHead className="text-right">Action</TableHead>
                    </TableRow>
                </TableHeader>
                <TableBody>
                    {openTrades.map((trade) => (
                    <TableRow key={trade.id}>
                        <TableCell className="font-medium">{trade.symbol}</TableCell>
                        <TableCell>
                        <Badge variant={trade.type === 'Buy' ? 'default' : 'destructive'} className={trade.type === 'Buy' ? 'bg-blue-500 hover:bg-blue-600' : 'bg-red-500 hover:bg-red-600'}>
                            {trade.type}
                        </Badge>
                        </TableCell>
                        <TableCell>{trade.volume.toFixed(2)}</TableCell>
                        <TableCell>{formatPrice(trade.openPrice)}</TableCell>
                        <TableCell>{formatPrice(marketData[trade.symbol]?.price || 0)}</TableCell>
                        <TableCell className={`text-right font-medium ${trade.profit >= 0 ? 'text-green-600' : 'text-red-600'}`}>
                          {formatCurrency(trade.profit)}
                        </TableCell>
                         <TableCell className="text-right">
                          <Button size="sm" variant="outline" onClick={() => simulator.closeTrade(trade.id)}>
                            Close
                          </Button>
                        </TableCell>
                    </TableRow>
                    ))}
                     {openTrades.length === 0 && (
                        <TableRow>
                            <TableCell colSpan={7} className="text-center text-muted-foreground">
                                No open positions.
                            </TableCell>
                        </TableRow>
                    )}
                </TableBody>
                </Table>
            </ScrollArea>
          </CardContent>
        </Card>
      </div>
      <div>
        <Card>
          <CardHeader>
            <CardTitle>Market Watch</CardTitle>
            <CardDescription>Live prices from the simulator.</CardDescription>
          </CardHeader>
          <CardContent>
            <Table>
                <TableHeader>
                    <TableRow>
                        <TableHead>Symbol</TableHead>
                        <TableHead className="text-right">Price</TableHead>
                        <TableHead className="text-right">Action</TableHead>
                    </TableRow>
                </TableHeader>
                <TableBody>
                    {Object.entries(marketData).map(([symbol, data]) => (
                        <TableRow key={symbol}>
                            <TableCell className="font-medium">{symbol}</TableCell>
                            <TableCell className="text-right">{formatPrice(data.price)}</TableCell>
                            <TableCell className="text-right space-x-1">
                                <Button size="sm" className="bg-blue-500 hover:bg-blue-600" onClick={() => simulator.openTrade(symbol, 'Buy', 0.1)}>Buy</Button>
                                <Button size="sm" className="bg-red-500 hover:bg-red-600" onClick={() => simulator.openTrade(symbol, 'Sell', 0.1)}>Sell</Button>
                            </TableCell>
                        </TableRow>
                    ))}
                </TableBody>
            </Table>
          </CardContent>
        </Card>
      </div>
    </div>
  );
}
