
'use client';

import {
  Card,
  CardContent,
  CardFooter,
  CardHeader,
  CardTitle,
} from '@/components/ui/card';
import { Button } from '@/components/ui/button';
import { Badge } from '@/components/ui/badge';
import type { ContentItem } from '@/lib/types';
import Image from 'next/image';
import Link from 'next/link';
import { FileText, Video, Music, Image as ImageIcon, Code, User, ArrowRight } from 'lucide-react';

interface ContentCardProps {
  item: ContentItem;
}

export function ContentCard({ item }: ContentCardProps) {
  const formatCurrency = (value: number) =>
    new Intl.NumberFormat('en-TZ', { style: 'currency', currency: 'TZS', minimumFractionDigits: 0 }).format(value);

  const TypeIcon = () => {
    switch (item.type) {
      case 'pdf': return <FileText className="h-4 w-4" />;
      case 'video': return <Video className="h-4 w-4" />;
      case 'audio': return <Music className="h-4 w-4" />;
      case 'image': return <ImageIcon className="h-4 w-4" />;
      case 'software': return <Code className="h-4 w-4" />;
      default: return <FileText className="h-4 w-4" />;
    }
  };

  return (
    <Card className="flex flex-col overflow-hidden group hover:shadow-xl transition-all duration-300 border-none bg-card/60 backdrop-blur-sm">
      <div className="relative aspect-video overflow-hidden">
        <Image 
          src={item.thumbnail} 
          alt={item.title} 
          fill 
          className="object-cover group-hover:scale-105 transition-transform duration-500"
          data-ai-hint="content thumbnail"
        />
        <div className="absolute top-3 right-3">
          <Badge className="bg-white/90 text-black hover:bg-white flex gap-1.5 items-center backdrop-blur-md border-none">
            <TypeIcon />
            {item.type.toUpperCase()}
          </Badge>
        </div>
      </div>
      <CardHeader className="pb-2">
        <div className="flex items-center gap-2 text-xs text-muted-foreground mb-1">
            <User className="h-3 w-3" />
            <span>{item.creatorName}</span>
        </div>
        <CardTitle className="font-headline text-lg line-clamp-1 leading-tight">{item.title}</CardTitle>
      </CardHeader>
      <CardContent className="flex-grow">
        <p className="text-sm text-muted-foreground line-clamp-2">{item.description}</p>
      </CardContent>
      <CardFooter className="flex items-center justify-between pt-0">
        <div className="font-bold text-primary text-lg">{formatCurrency(item.price)}</div>
        <Link href={`/content/${item.id}`} passHref>
            <Button size="sm" className="rounded-xl group-hover:translate-x-1 transition-transform">
                Lipia <ArrowRight className="ml-2 h-4 w-4" />
            </Button>
        </Link>
      </CardFooter>
    </Card>
  );
}
