'use client';

import { useState, useEffect, useCallback } from 'react';
import { useNotifications } from '@/contexts/notification-context';
import { checkChallengeRules } from '@/lib/actions';

import { DashboardOverview } from '@/components/dashboard-overview';
import { ChallengeOverview } from '@/components/challenge-overview';
import { EquityChart } from '@/components/equity-chart';
import { TradeHistory } from '@/components/trade-history';
import { challenges } from '@/lib/data';
import { TradeSimulator } from '@/services/trade-simulator';
import type { Trade } from '@/lib/types';

const challenge = challenges[0];

export function ActiveChallengeWrapper() {
  const { addNotification } = useNotifications();
  const [simulator] = useState(() => TradeSimulator.getInstance(challenge.startCapital));
  
  const [accountState, setAccountState] = useState(simulator.getAccountState());
  const [equityData, setEquityData] = useState<{ date: string; equity: number }[]>([]);
  const [trades, setTrades] = useState<Trade[]>([]);
  
  const handleTick = useCallback(() => {
    const state = simulator.getAccountState();
    setAccountState(state);
    setTrades(simulator.getTradeHistory());

    const today = new Date().toISOString().slice(0, 10);
    setEquityData(prevData => {
        const lastEntry = prevData[prevData.length - 1];
        if (lastEntry?.date === today) {
            // To avoid too many points, we can update last point instead of adding new one frequently
            if (prevData.length > 1 && (Date.now() - new Date(lastEntry.date).getTime()) < 5000 ) {
                 return [...prevData.slice(0, -1), { date: today, equity: state.equity }];
            }
        }
        if (prevData.length > 100) {
            return [...prevData.slice(1), { date: today, equity: state.equity }];
        }
        return [...prevData, { date: today, equity: state.equity }];
    });
    
    checkChallengeRules({
      balance: state.balance,
      profit: state.profit,
      dailyDrawdown: state.dailyDrawdown,
      maxDrawdown: state.maxDrawdown,
      profitTarget: challenge.profitTarget,
      dailyDrawdownLimit: challenge.dailyDrawdownLimit,
      maxDrawdownLimit: challenge.maxDrawdownLimit,
    }).then(result => {
      if (result.isViolatingRules && result.notificationMessage) {
          const lastNotif = simulator.getLastNotification();
          if (lastNotif?.description !== result.notificationMessage) {
              addNotification({
                  title: 'Rule Violation Alert',
                  description: result.notificationMessage,
                  variant: 'destructive',
              });
              simulator.setLastNotification({
                  title: 'Rule Violation Alert',
                  description: result.notificationMessage,
              });
          }
      } else if(state.profit >= challenge.profitTarget) {
          const lastNotif = simulator.getLastNotification();
          if (lastNotif?.title !== 'Congratulations!') {
              const message = `You have reached the profit target of ${challenge.profitTarget}. You can now request a payout.`;
              addNotification({
                  title: 'Congratulations!',
                  description: message,
              });
              simulator.setLastNotification({
                  title: 'Congratulations!',
                  description: message,
              });
         }
      }
    });
  }, [addNotification, simulator]);

  useEffect(() => {
    simulator.on('tick', handleTick);
    simulator.start();

    // Set initial data
    setAccountState(simulator.getAccountState());
    setEquityData([{ date: new Date().toISOString().slice(0,10), equity: simulator.getAccountState().equity }]);
    setTrades(simulator.getTradeHistory());

    return () => {
      simulator.off('tick', handleTick);
    };
  }, [simulator, handleTick]);

  return (
    <>
        <DashboardOverview 
            balance={accountState.balance}
            equity={accountState.equity}
            profit={accountState.profit}
        />
        <div className="grid gap-4 md:grid-cols-2 lg:grid-cols-7">
            <div className="lg:col-span-4 space-y-4">
                <EquityChart data={equityData} />
            </div>
            <div className="lg:col-span-3 space-y-4">
                <ChallengeOverview
                    profitTarget={challenge.profitTarget}
                    maxDrawdown={challenge.maxDrawdownLimit}
                    dailyDrawdown={challenge.dailyDrawdownLimit}
                    currentProfit={accountState.profit}
                    currentMaxDrawdown={accountState.maxDrawdown}
                    currentDailyDrawdown={accountState.dailyDrawdown}
                />
                <TradeHistory trades={trades.slice(0, 5)} />
            </div>
        </div>
    </>
  );
}
