
'use client';

import { useUser, useCollection, useFirestore } from '@/firebase';
import { query, collection, where, orderBy } from 'firebase/firestore';
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '@/components/ui/card';
import { Button } from '@/components/ui/button';
import { Badge } from '@/components/ui/badge';
import { Wallet, ArrowUpRight, ArrowDownLeft, History } from 'lucide-react';
import { Skeleton } from '@/components/ui/skeleton';

export default function WalletPage() {
  const { user } = useUser();
  const db = useFirestore();

  const transactionsQuery = user && db 
    ? query(collection(db, 'transactions'), where('userId', '==', user.uid), orderBy('timestamp', 'desc'))
    : null;

  const { data: transactions, loading } = useCollection(transactionsQuery);

  const formatCurrency = (value: number) =>
    new Intl.NumberFormat('en-TZ', { style: 'currency', currency: 'TZS', minimumFractionDigits: 0 }).format(value);

  const balance = transactions?.reduce((acc, curr: any) => acc + curr.amount, 0) || 0;

  return (
    <div className="space-y-8">
      <div className="flex flex-col md:flex-row md:items-center justify-between gap-4">
        <div>
          <h1 className="text-3xl font-bold font-headline">My Wallet</h1>
          <p className="text-muted-foreground">Manage your earnings and payment history.</p>
        </div>
        <div className="flex gap-2">
           <Button variant="outline" className="rounded-2xl h-12 px-6">Top Up</Button>
           <Button className="rounded-2xl h-12 px-6">Withdraw</Button>
        </div>
      </div>

      <div className="grid grid-cols-1 lg:grid-cols-3 gap-6">
        <Card className="bg-primary text-primary-foreground rounded-3xl border-none shadow-2xl shadow-primary/20">
          <CardHeader>
            <CardTitle className="text-primary-foreground/80 font-medium text-sm">Available Balance</CardTitle>
          </CardHeader>
          <CardContent className="space-y-4">
            <div className="text-5xl font-bold font-headline tracking-tighter">
              {formatCurrency(balance)}
            </div>
            <div className="flex items-center gap-2 text-sm text-primary-foreground/70">
              <Wallet className="h-4 w-4" />
              Secure Tanzanian Shillings
            </div>
          </CardContent>
        </Card>

        <Card className="lg:col-span-2 rounded-3xl border-none bg-card/60 backdrop-blur-sm">
          <CardHeader className="flex flex-row items-center justify-between">
            <div>
              <CardTitle className="font-headline">Recent Activity</CardTitle>
              <CardDescription>All your mobile money movements.</CardDescription>
            </div>
            <History className="h-5 w-5 text-muted-foreground" />
          </CardHeader>
          <CardContent>
            {loading ? (
              <div className="space-y-4">
                {[1, 2, 3].map(i => <Skeleton key={i} className="h-16 w-full rounded-2xl" />)}
              </div>
            ) : !transactions || transactions.length === 0 ? (
              <div className="py-20 text-center text-muted-foreground">
                <p>No transactions yet.</p>
              </div>
            ) : (
              <div className="space-y-6">
                {transactions.map((tx: any) => (
                  <div key={tx.id} className="flex items-center justify-between">
                    <div className="flex items-center gap-4">
                      <div className={`h-10 w-10 rounded-xl flex items-center justify-center ${
                        tx.amount > 0 ? 'bg-green-100 text-green-600' : 'bg-red-100 text-red-600'
                      }`}>
                        {tx.amount > 0 ? <ArrowUpRight className="h-5 w-5" /> : <ArrowDownLeft className="h-5 w-5" />}
                      </div>
                      <div>
                        <p className="font-bold text-sm">{tx.description || 'Lipia/ Transaction'}</p>
                        <p className="text-xs text-muted-foreground">
                            {tx.timestamp?.toDate ? tx.timestamp.toDate().toLocaleString() : 'Just now'}
                        </p>
                      </div>
                    </div>
                    <div className="text-right">
                      <p className={`font-bold ${tx.amount > 0 ? 'text-green-600' : 'text-foreground'}`}>
                        {tx.amount > 0 ? '+' : ''}{formatCurrency(tx.amount)}
                      </p>
                      <Badge variant="outline" className="text-[10px] uppercase">{tx.status}</Badge>
                    </div>
                  </div>
                ))}
              </div>
            )}
          </CardContent>
        </Card>
      </div>
    </div>
  );
}
