
'use client';

import { useCollection, useFirestore } from '@/firebase';
import { collection, query, orderBy } from 'firebase/firestore';
import { ContentCard } from '@/components/content-card';
import { Input } from '@/components/ui/input';
import { Search, Loader2 } from 'lucide-react';
import { useState, useMemo } from 'react';
import { Button } from '@/components/ui/button';
import { Skeleton } from '@/components/ui/skeleton';

export default function ExplorePage() {
  const db = useFirestore();
  const [search, setSearch] = useState('');
  const [activeFilter, setActiveFilter] = useState('all');

  const contentQuery = useMemo(() => {
    if (!db) return null;
    return query(collection(db, 'content'), orderBy('createdAt', 'desc'));
  }, [db]);

  const { data: content, loading } = useCollection(contentQuery);

  const filteredItems = useMemo(() => {
    if (!content) return [];
    return content.filter(item => {
      const matchesSearch = item.title.toLowerCase().includes(search.toLowerCase());
      const matchesFilter = activeFilter === 'all' || item.type === activeFilter;
      return matchesSearch && matchesFilter;
    });
  }, [content, search, activeFilter]);

  const categories = [
    { label: 'All', value: 'all' },
    { label: 'Video', value: 'video' },
    { label: 'PDFs', value: 'pdf' },
    { label: 'Images', value: 'image' },
    { label: 'Software', value: 'software' },
    { label: 'Audio', value: 'audio' },
  ];

  return (
    <div className="space-y-8">
        <div className="flex flex-col md:flex-row md:items-end justify-between gap-4">
            <div>
                <h1 className="text-3xl font-bold font-headline">Explore Marketplace</h1>
                <p className="text-muted-foreground">Discover premium digital content from top creators in East Africa.</p>
            </div>
            <div className="relative w-full md:w-80">
                <Search className="absolute left-3 top-1/2 -translate-y-1/2 h-4 w-4 text-muted-foreground" />
                <Input 
                    placeholder="Search content..." 
                    className="pl-10 rounded-xl" 
                    value={search}
                    onChange={(e) => setSearch(e.target.value)}
                />
            </div>
        </div>

        <div className="flex gap-2 overflow-x-auto pb-2 scrollbar-hide">
            {categories.map((cat) => (
                <Button 
                    key={cat.value} 
                    variant={activeFilter === cat.value ? "default" : "outline"} 
                    size="sm"
                    className="rounded-full px-5"
                    onClick={() => setActiveFilter(cat.value)}
                >
                    {cat.label}
                </Button>
            ))}
        </div>

      {loading ? (
        <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 xl:grid-cols-4 gap-6">
            {[1, 2, 3, 4].map(i => <Skeleton key={i} className="aspect-video w-full rounded-2xl" />)}
        </div>
      ) : (
        <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 xl:grid-cols-4 gap-6">
          {filteredItems.map((item) => (
            <ContentCard key={item.id} item={item as any} />
          ))}
        </div>
      )}
      
      {!loading && filteredItems.length === 0 && (
          <div className="py-20 text-center border-2 border-dashed rounded-3xl">
              <p className="text-muted-foreground">No content found matching your criteria.</p>
              <Button variant="link" onClick={() => { setSearch(''); setActiveFilter('all'); }}>Clear all filters</Button>
          </div>
      )}
    </div>
  );
}
