// Use server directive is required for Genkit Flows.
'use server';

/**
 * @fileOverview A flow to generate culturally relevant Swahili names for trading challenges.
 *
 * - generateSwahiliChallengeNames - A function that generates Swahili challenge names.
 * - GenerateSwahiliChallengeNamesInput - The input type for the generateSwahiliChallengeNames function.
 * - GenerateSwahiliChallengeNamesOutput - The return type for the generateSwahiliChallengeNames function.
 */

import {ai} from '@/ai/genkit';
import {z} from 'genkit';

const GenerateSwahiliChallengeNamesInputSchema = z.object({
  challengeType: z
    .string()
    .describe('The type of trading challenge (e.g., Instant Funding, 1-Step, 2-Step).'),
});
export type GenerateSwahiliChallengeNamesInput = z.infer<
  typeof GenerateSwahiliChallengeNamesInputSchema
>;

const GenerateSwahiliChallengeNamesOutputSchema = z.object({
  challengeName: z
    .string()
    .describe('A culturally relevant Swahili name for the trading challenge.'),
});
export type GenerateSwahiliChallengeNamesOutput = z.infer<
  typeof GenerateSwahiliChallengeNamesOutputSchema
>;

export async function generateSwahiliChallengeNames(
  input: GenerateSwahiliChallengeNamesInput
): Promise<GenerateSwahiliChallengeNamesOutput> {
  return generateSwahiliChallengeNamesFlow(input);
}

const prompt = ai.definePrompt({
  name: 'generateSwahiliChallengeNamesPrompt',
  input: {schema: GenerateSwahiliChallengeNamesInputSchema},
  output: {schema: GenerateSwahiliChallengeNamesOutputSchema},
  prompt: `You are a creative marketing specialist with expertise in Swahili culture.

You are tasked with generating a culturally relevant and engaging Swahili name for a trading challenge.

The trading challenge type is: {{{challengeType}}}

Consider incorporating elements of Tanzanian culture, proverbs, or popular Swahili phrases to make the name appealing to the target audience.

Provide only the name of the challenge in Swahili.
`,
});

const generateSwahiliChallengeNamesFlow = ai.defineFlow(
  {
    name: 'generateSwahiliChallengeNamesFlow',
    inputSchema: GenerateSwahiliChallengeNamesInputSchema,
    outputSchema: GenerateSwahiliChallengeNamesOutputSchema,
  },
  async input => {
    const {output} = await prompt(input);
    return output!;
  }
);
