'use server';

/**
 * @fileOverview Enforces trading challenge rules using an LLM to provide
 * notifications if the user is violating challenge rules.
 *
 * - enforceChallengeRules - A function that enforces the challenge rules.
 * - EnforceChallengeRulesInput - The input type for the enforceChallengeRules function.
 * - EnforceChallengeRulesOutput - The return type for the enforceChallengeRules function.
 */

import {ai} from '@/ai/genkit';
import {z} from 'genkit';

const EnforceChallengeRulesInputSchema = z.object({
  balance: z.number().describe('The current balance of the trading account.'),
  profit: z.number().describe('The current profit of the trading account.'),
  dailyDrawdown: z
    .number()
    .describe('The current daily drawdown of the trading account.'),
  maxDrawdown: z
    .number()
    .describe('The maximum drawdown of the trading account.'),
  profitTarget: z.number().describe('The profit target for the challenge.'),
  dailyDrawdownLimit: z
    .number()
    .describe('The daily drawdown limit for the challenge.'),
  maxDrawdownLimit: z
    .number()
    .describe('The maximum drawdown limit for the challenge.'),
});
export type EnforceChallengeRulesInput = z.infer<
  typeof EnforceChallengeRulesInputSchema
>;

const EnforceChallengeRulesOutputSchema = z.object({
  isViolatingRules: z
    .boolean()
    .describe('Whether the trader is violating any challenge rules.'),
  notificationMessage: z
    .string()
    .describe(
      'A message to the trader indicating which rule they are violating, or null if not violating any rules.'
    ),
});
export type EnforceChallengeRulesOutput = z.infer<
  typeof EnforceChallengeRulesOutputSchema
>;

export async function enforceChallengeRules(
  input: EnforceChallengeRulesInput
): Promise<EnforceChallengeRulesOutput> {
  return enforceChallengeRulesFlow(input);
}

const prompt = ai.definePrompt({
  name: 'enforceChallengeRulesPrompt',
  input: {schema: EnforceChallengeRulesInputSchema},
  output: {schema: EnforceChallengeRulesOutputSchema},
  prompt: `You are a trading challenge rules enforcer. You will review the
  trader's account and determine if they are violating any of the challenge
  rules. The challenge rules are as follows:

  - Daily Drawdown Limit: {{dailyDrawdownLimit}}
  - Maximum Drawdown Limit: {{maxDrawdownLimit}}
  - Profit Target: {{profitTarget}}

  Here is the trader's current account information:

  - Balance: {{balance}}
  - Profit: {{profit}}
  - Daily Drawdown: {{dailyDrawdown}}
  - Maximum Drawdown: {{maxDrawdown}}

  Based on this information, determine if the trader is violating any of the
  challenge rules. If they are, set isViolatingRules to true and provide a
  clear and concise notification message indicating which rule they are
  violating. If they are not violating any rules, set isViolatingRules to
  false and set notificationMessage to null.

  Consider each rule separately and independently. For example, even if the
  trader has exceeded the maximum drawdown limit, you should still check if
  they have also exceeded the daily drawdown limit.

  Ensure that notificationMessage is null if and only if isViolatingRules is
  false.

  {{~#if (gt dailyDrawdown dailyDrawdownLimit)~}}
  The trader is violating the daily drawdown limit.
  {{~/if~}}

  {{~#if (gt maxDrawdown maxDrawdownLimit)~}}
  The trader is violating the maximum drawdown limit.
  {{~/if~}}

  {{~#if (lt profit profitTarget)~}}
  The trader has not yet reached the profit target.
  {{~/if~}}`,
});

const enforceChallengeRulesFlow = ai.defineFlow(
  {
    name: 'enforceChallengeRulesFlow',
    inputSchema: EnforceChallengeRulesInputSchema,
    outputSchema: EnforceChallengeRulesOutputSchema,
  },
  async input => {
    const {output} = await prompt(input);
    return output!;
  }
);
